%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code:  Example 5.1 
% File: ENSO3.m  (F^{(2,3)}_{T} test statistic) 
%
% Programmed by  Bruce E. Hansen.
% For updates and contact information, see the webpage:
% www.ssc.wisc.edu/~bhansen
%
% This program estimates a two-regime SETAR model, and tests the null  
% of a two-regime SETAR against the alternative of a three-regime SETAR. 
%
% Reference:
% Hansen, B.E. (1999). Testing for linearity.
%   Journal of Economic Surveys, 13(5), 551-576.
%   DOI: 10.1111/1467-6419.00098.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function ENSO3
load ENSOdat.txt;
dat=ENSOdat;  

global p;p=5;		        % Order of autoregression 
global n;
global dmin_;dmin_=1;		% Minimal delay order 
global dmax_;dmax_=p;		% Maximal delay order (set to p or less)
global trim_;trim_ = .1;	% Minimal percentage of data per regime 
global qnum_;qnum_=100;	        % Number of thresholds to search. 
                                % Set qnum_=0 to search over all values 
global boot_;boot_ = 1000;	% Bootstrap replications 
global k;
global t;

% Define data 
n = length(dat(:,1));
y = dat(p+1:n);
t = n-p;
x = [ones(t,1),dat(p:n-1)];
xname  = ['Constant';' Y(t-01)'];
xname2 = ['Constant  ';' Y(t-01)^2'];
j=2;
while j<=p
    x=[x,dat(p+1-j:n-j)];
    if j<10
        kname = strcat('_Y(t-0',num2str(j),')');
    else
        kname = strcat('_Y(t-',num2str(j),')');
    end;
    xname  = [xname;kname];
    xname2 = xname;
    j=j+1;
end;
k = length(x(1,:));

% Linear Regression 
mi   = inv(x'*x);
beta = mi*(x'*y);
e    = y-x*beta;
ee   = e'*e;
xe   = x.*(e*ones(1,length(x(1,:))));
sig  = ee/t;
se   = sqrt(diag(mi*(xe'*xe)*mi));

% Conditional Variance
e2 = e.*e;
x2 = x.*x;
m2 = inv(x2'*x2);
hetbeta = m2*(x2'*e2);
h   = x2*hetbeta;
eh  = e./sqrt(h.*(h>0)+(h<0)).*(h>0);
u   = e2-h;
x2u = x2.*(u*ones(1,length(x(1,:))));
se2 = sqrt(diag(m2*(x2u'*x2u)*m2));
em  = e2-mean(e2)';
fh  = t*((em'*em)/(u'*u)-1);

% Report Linear Estimation %
fprintf('Dependent Variable:   %s\n',yname);
disp(' ');
disp('Linear Autoregression');
disp(' ');
hname=['Variable     ','Estimate     ','St Error'];
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(beta)
    fprintf('%s     %f    %f\n',xname(i,:),beta(i),se(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',t);
fprintf('Sum of Squared Errors:         %f\n',ee);
fprintf('Residual Variance:             %f\n',sig);
disp(' ');
disp(' ');
disp('Conditional Variance');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(hetbeta)
    fprintf('%s     %f    %f\n',xname2(i,:),hetbeta(i),se2(i));
end;
fprintf('Heteroskedasticity F Test:     %f      %f\n',fh,1-chi2cdf(fh,p));
disp('----------------------------------');
disp(' ');
disp(' ');
disp(' ');
disp(' ');
% TAR Estimation %
[f,dhat,ghat,ghat1,ghat2] = tar2(dat);
q     = qfunc(x);
qd    = q(:,dhat);
delay = dmin_-1+dhat;

% SETAR_1 Model 
d1   = (qd<=ghat);
d2   = 1-d1;
x1   = x.*(d1*ones(1,length(x(1,:))));
x2   = x.*(d2*ones(1,length(x(1,:))));
xx   = [x1,x2];
mxx  = inv(xx'*xx);
beta = mxx*(xx'*y);
e    = y-xx*beta;
xxe  = xx.*(e*ones(1,length(xx(1,:))));
ee   = e'*e;
sig  = ee/t;

n1   = sum(d1);
n2   = sum(d2);
se   = sqrt(diag(mxx*(xxe'*xxe)*mxx));
b1   = beta(1:k);
b2   = beta(k+1:2*k);
sig1 = (e.^2)'*d1/n1;
sig2 = (e.^2)'*d2/n2;

yz    = e.*e;
d12   = [d1,d2];
z     = [d12,x(:,2:k).*x(:,2:k)];
mz    = inv(z'*z);
betaz = mz*(z'*yz);
h     = z*betaz;
u     = yz-h;
zu    = z.*(u*ones(1,length(z(1,:))));
se3   = sqrt(diag(mz*(zu'*zu)*mz));
em    = yz-mean(yz)';
fz    = t*((em'*em)/(u'*u)-1);
ez    = e./sqrt(h.*(h>0)+(h<=0)).*(h>0);

hname=['Variable     ','Estimate     ','St Error'];
fprintf('Dependent Variable:    %s\n',yname)
disp(' ');
disp('Threshold Autoregression, Single Threshold');
fprintf('Sum of Squared Errors:         %f\n',ee);
fprintf('Residual Variance:             %f\n',sig);
fprintf('Delay Order:                   %u\n',delay);
fprintf('Threshold Estimate:            %f\n',ghat);
disp(' ');
disp(' ');

disp('Regime 1 (Threshold Variable less than or equal to Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:k
    fprintf('%s     %f    %f\n',xname(i,:),b1(i),se(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n1);
fprintf('Percentage:                    %f\n',n1/t);
fprintf('Regime Variance:               %f\n',sig1);
disp(' ');disp(' ');

disp('Regime 2 (Threshold Variable greater than Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=k+1:2*k
    fprintf('%s     %f    %f\n',xname(i-k,:),b2(i-k),se(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n2);
fprintf('Percentage:                    %f\n',n2/t);
fprintf('Regime Variance:               %f\n',sig2);
disp(' ');disp(' ');disp(' ');

xname3=['      D1';'      D2';xname2(2:k,:)];
disp('Conditional Variance');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(betaz)
    fprintf('%s     %f    %f\n',xname3(i,:),betaz(i),se3(i));
end;
fprintf('Heteroskedasticity F Test:     %f      %f\n',fz,1-chi2cdf(fz,k));
disp('----------------------------------');
disp(' ');
disp(' ');
disp(' ');
disp(' ');

% SETAR_2 Model 
d1   = (qd<=ghat1);
d2   = (qd<=ghat2).*(1-d1);
d3   = 1-d1-d2;
x1   = x.*(d1*ones(1,length(x(1,:))));
x2   = x.*(d2*ones(1,length(x(1,:))));
x3   = x.*(d3*ones(1,length(x(1,:))));
xx   = [x1,x2,x3];
mxx  = inv(xx'*xx);

betatar = mxx*(xx'*y);
etar    = y-xx*betatar;
xxe     = xx.*(etar*ones(1,length(xx(1,:))));
eetar   = etar'*etar;
sigtar  = eetar/t;

n1 = sum(d1);
n2 = sum(d2);
n3 = sum(d3);
setar = sqrt(diag(mxx*(xxe'*xxe)*mxx));

sig1 = (etar.^2)'*d1/n1;
sig2 = (etar.^2)'*d2/n2;
sig3 = (etar.^2)'*d3/n3;
temp = 0;
for i=1:length(d1)
    if d1(i)==1;
        if temp==0
            e1   = d1(i);
            temp = 1;
        else
            e1 = [e1;d1(i)];
        end;
    end;
end;
temp=0;
for i=1:length(d2)
    if d2(i)==1;
        if temp==0
            e2   = d2(i);
            temp = 1;
        else
            e2 = [e1;d2(i)];
        end;
    end;
end;

disp('Threshold Autoregression, Double Threshold');
fprintf('Sum of Squared Errors:         %f\n',eetar);
fprintf('Residual Variance:             %f\n',sigtar);
fprintf('Delay Order:                   %u\n',delay);
fprintf('First Threshold Estimate:      %f\n',ghat1);
fprintf('Second Threshold Estimate:     %f\n',ghat2);
disp(' ');disp(' ');
disp('Regime 1 (Threshold Variable less than or equal to First Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:k
    fprintf('%s     %f    %f\n',xname(i,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n1);
fprintf('Percentage:                    %f\n',n1/t);
fprintf('Regime Variance:              %f\n',sig1);
disp(' ');disp(' ');

disp('Regime 2 (Threshold Variable Between Thresholds):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=k+1:2*k
    fprintf('%s     %f    %f\n',xname(i-k,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n2);
fprintf('Percentage:                    %f\n',n2/t);
fprintf('Regime Variance:               %f\n',sig2);
disp(' ');disp(' ');disp(' ');

disp('Regime 3 (Threshold Variable Above Second Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=(2*k+1):(3*k)
    fprintf('%s     %f    %f\n',xname(i-2*k,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n3);
fprintf('Percentage:                    %f\n',n3/t);
fprintf('Regime Variance:               %f\n',sig3);
disp(' ');disp(' ');disp(' ');

y0=dat(1:p);

% Model-Based Bootstrap, Homoskedastic Errors 
fb1 = zeros(boot_,1);
i=1;
while i<=boot_
    [fbi,dhatb,ghatb,ghat1b,ghat2b] = tar2(tar_sim1(y0,b1,b2,delay,ghat,e));
    fb1(i) = fbi;
    i=i+1;
end;
pv1 = mean(fb1>(ones(length(fb1(:,1)),1)*f))';

% Model-Based Bootstrap, Regime-Specific Errors %
fb2 = zeros(boot_,1);
i=1;
while i<=boot_
    [fbi,dhatb,ghatb,ghat1b,ghat2b] = tar2(tar_sim2(y0,b1,b2,delay,ghat,e1,e2));
    fb2(i) = fbi;
    i=i+1;
end;
pv2 = mean(fb2>(ones(length(fb2(:,1)),1)*f))';

% Model-Based Bootstrap, Regime-Specific Errors %
fb3 = zeros(boot_,1);
i=1;
while i<=boot_
    [fbi,dhatb,ghatb,ghat1b,ghat2b] = tar2(tar_sim2(y0,b1,b2,delay,ghat,betaz,ez));
    fb3(i) = fbi;
    i=i+1;
end;
pv3 = mean(fb3>(ones(length(fb3(:,1)),1)*f))';

% Output %
fprintf('Bootstrap Replications:          %u\n',boot_);    
disp(' ');
disp(' ');
disp('TAR Tests, P-Values');
disp(' ');
disp('Fstat      PV1     PV2     PV2');
disp('------------------------------');
for i=1:length(pv1)
fprintf('%f   %f   %f   %f\n',f(i),pv1(i),pv2(i),pv3(i));
end;
disp(' ');
disp(' ');


% Density Calculation 
ub     = max([quant(fb1,.99);quant(fb2,.99);quant(fb3,.99)])';
k_x    = (0:ub/999:ub)';
k_chi  = (k_x.^(k/2-1)).*exp(-k_x/2)./(gamma(k/2)*(2^(k/2)));
kern   = [k_chi,kernel(fb1,k_x),kernel(fb2,k_x),kernel(fb3,k_x)];
sun_k3 = [k_x,kern];
save('sun_k3.mat','sun_k3');

% TAR Procedures %
function [dhat,ghat]=tar(y,x,q,qq)
mi = ginv(x'*x);
e  = y-x*mi*(x'*y);
qn = length(q(1,:));
gn = length(qq(:,1));
s  = zeros(gn,qn);
m=1;
while m<=qn
    qm = q(:,m);
    j=1;
    while j<=gn
        gg = qq(j,m);
        if isnan(gg)
            continue;
        end;
        xd  = x.*((qm<=gg)*ones(1,length(x(1,:))));
        xxd = xd'*xd;
        mmi = xxd-xxd*mi*xxd;
        xde = xd'*e;
        s(j,m) = xde'*dinv(xde,mmi);
        j=j+1;
    end;
    m=m+1;
end;
[temp,dhat] = max(max(s)');
[temp,kkk]  = max(s(:,dhat));
ghat        = qq(kkk,dhat);
clear temp;
clear kkk;

function [ghat]=tar_d(y,x,x1,q,qq)
mi =ginv(x'*x);
e  = y-x*mi*(x'*y);
gn = length(qq(:,1));
s  = zeros(gn,1);
j=1;
while j<=gn;
    gg = qq(j);
    if isnan(gg)
       continue;
    end;
    xd   = [x1,x.*((q<=gg)*ones(1,length(x(1,:))))];
    xxd  = xd'*xd;
    xdx  = x'*xd;
    mmi  = xxd-xdx'*mi*xdx;
    xde  = xd'*e;
    s(j) = xde'*dinv(xde,mmi);
    j=j+1;
end;
[f,temp] = max(s);
ghat     = qq(temp);
clear temp;

function qq=qsort(q)
global trim_;
global qnum_;
n  = length(q(:,1));
k  = length(q(1,:));
n1 = round(trim_*n);
if qnum_==0;
    qq = zeros(n-n1-n1+1,k);
    qq = nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j),1);
        qj = qj(n1:(length(qj(:,1))-n1+1));
        qq(1:length(qj(:,1)),j) = qj;
        j=j+1;
    end;
else
    qq = zeros(qnum_,k);
    qq = nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j));
        qj = qj(n1:(length(qj(:,1))-n1+1));
        nj = length(qj(:,1));
        if nj<=qnum_
            qq(1:nj,j) = qj;
        else
            qq(:,j) = qj(ceil((1:qnum_)./qnum_*nj));
        end;
        j=j+1;
    end;
end;

function qq=qsort_2(q,ghat)
global trim_;
global qnum_;
n  = length(q(:,1));
n1 = round(trim_*n);
qq = unique(q);
qq = qq(n1:(length(qq(:,1))-n1+1));

[temp,kkk] = max(qq>=ghat);
ddd = abs((1:length(qq(:,1)))-kkk)<n1;
temp=0;
for i=1:length(ddd)
    if ddd(i)==0
        if temp==0
            kkk = qq(i,:);
            temp = 1;
        else
            kkk = [kkk;qq(i,:)];
        end;
    end;
end;
qq=kkk;
clear temp;
clear ddd;
clear kkk;
if qnum_>0
    nq = length(qq(:,1));
    if nq>qnum_
        qq = qq(ceil((1:qnum_)'./qnum_*nq));
    end;
end;

function q=qfunc(x)
global dmin_;
global dmax_;
q = x(:,(dmin_+1):(dmax_+1));

function [f,dhat,ghat,ghat1,ghat2]=tar2(dat)
global p;
global n;
global t;
y = dat(p+1:n);
x = [ones(t,1),dat(p:n-1)];
j=2;
while j<=p
    x = [x,dat(p+1-j:n-j)];
    j=j+1;
end;
q = qfunc(x);
[dhat,ghat] = tar(y,x,q,qsort(q));           % fit TAR-1 model 
qd = q(:,dhat);
x1 = x.*((qd<=ghat)*ones(1,length(x(1,:))));
xx = [x1,x];
e  = y-xx*(y'/xx')';
ghat2 = tar_d(y,x,x1,qd,qsort_2(qd,ghat));   % fit TAR-2 model 
x1    = x.*((qd<=ghat2)*ones(1,length(x(1,:))));
ghat1 = tar_d(y,x,x1,qd,qsort_2(qd,ghat2));  % iterated estimate of gamma_1 
x1    = x.*((qd<=ghat1)*ones(1,length(x(1,:))));
ghat2 = tar_d(y,x,x1,qd,qsort_2(qd,ghat1));  % iterated estimate of gamma_2 
if ghat2<ghat1; 
    gg    = ghat1;
    ghat1 = ghat2;
    ghat2 = gg;
end;
d1   = (qd<=ghat1);
d2   = (qd<=ghat2).*(1-d1);
xx   = [(x.*(d1*ones(1,length(x(1,:))))),(x.*(d2*ones(1,length(x(1,:))))),(x.*((1-d1-d2)*ones(1,length(x(1,:)))))];
etar = y-xx*(y'/xx')';
f    = t*((e'*e)/(etar'*etar)-1);



%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation Procedures 
%%%%%%%%%%%%%%%%%%%%%%%%%
function y=tar_sim1(y0,b1,b2,delay,g,e)
global k;
global t;
global p;
u = e(ceil(unifrnd(0,1,t,1)*t));
y = [y0;zeros(t,1)];
j=1;
while j<=t
    x = y(p+j-1);
    for i=p+j-2:(-1):j
        x = [x;y(i)];
    end;
    q = x(delay);
    x = [1;x];
    y(p+j) = (b1'*x)*(q<=g)+(b2'*x)*(q>g)+u(j);
    j=j+1;
end;

function y=tar_sim2(y0,b1,b2,delay,g,e1,e2)
global t;
global p;
y  = [y0;zeros(t,1)];
u  = unifrnd(0,1,t,1);
u1 = e1(ceil(u*length(e1)));
u2 = e2(ceil(u*length(e2)));

j=1;
while j<=t
    x = y(p+j-1);
    for i=p+j-2:(-1):j
        x = [x;y(i)];
    end;
    q      = x(delay);
    xx     = [1;x];
    y(p+j) = (b1'*xx+u1(j))*(q<=g)+(b2'*xx+u2(j))*(q>g)+u(j);
    j=j+1;
end;
    
function y=tar_sim3(y0,b1,b2,delay,g,het,z)
global t;
global p;
y = [y0;zeros(t,1)];
u = z(ceil(unifrnd(0,1,t,1)*t));
j=1;
while j<=t
    x = y(p+j-1);
    for i=p+j-2:(-1):j
        x = [x;y(i)];
    end;
    q  =x(delay);
    xx = [1;x];
    d1 = (q<=g);
    d2 = 1-d1;
    h  = ([d1;d2;x].^2)'*het;
    y(p+j) = (b1'*xx)*d1+(b2'*xx)*d2+sqrt(h.*(h>0)).*u(j);
    j=j+1;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function kern=kernel(x,b)
h    = 1.7*std(x)'/(length(x(:,1))^(.2));
g    = length(b(:,1));
kern = zeros(g,1);
i=1;
while i<=g;
    u       = abs(b(i)-x)/h;
    kern(i) = mean((1-u.^2).*(u<=1))'*(.75)/h;
    i=i+1;
end;

function qq=quant(x,q)
s  = sortrows(x,1);
qq = s(round(length(s(:,1))*q));

function mi=ginv(m)
warning off;
lastwarn(' ');
mi = inv(m);
mw = ' ';
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   mi = pinv(m);
end;

function d=dinv(y,x)
warning off;
lastwarn(' ');
d=(y'/x')';
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   d = ginv(x'*x)*(x'*y);
end;
